import { useState } from "react";
import {
  TextField as MUITextField,
  FormControl,
  InputAdornment,
  IconButton,
  useTheme,
} from "@mui/material";
import {
  CheckCircle,
  ErrorRounded,
  Visibility,
  VisibilityOff,
} from "@mui/icons-material";

/**
 *
 * @param {String} label to display the label.
 * @param {SxProps<Theme>} sx The system prop that allows defining system overrides as well as additional CSS styles.
 * @returns {JSX.Element} Customized Text field
 */

export default function TextField({
  sx,
  label,
  hasValidationIcon,
  endAdornment,
  isPassword,
  maxLength,
  color,
  iscvvStyle,
  autoComplete,
  ...props
}) {
  const { palette } = useTheme();
  const [show, setShow] = useState(false);
  props.type = !isPassword ? props.type : show ? "text" : "password";
  const handleMouseDownPassword = (event) => {
    event.preventDefault();
  };
  return (
    <FormControl
      fullWidth
      sx={{
        ...sx,
        ".Mui-focused": {
          backgroundColor: `#FFFFFF`,
        },
        ".MuiFormLabel-root": {
          color: "#B3B9C8",
          top: "-2px",
          "&.MuiInputLabel-shrink": {
            top: "0px",

            fontSize: "12px",
            paddingLeft: "0px",
          },
        },
        ".MuiFormHelperText-root": {
          background: "transparent",
          marginLeft: "0px",
        },
        ".MuiFormLabel-asterisk": {
          color: "error.main",
        },
      }}
    >
      <MUITextField
        variant="filled"
        size="small"
        label={label}
        fullWidth
        inputProps={{
          maxLength: maxLength,
          autoComplete: autoComplete,
        }}
        InputProps={{
          endAdornment: hasValidationIcon ? (
            <InputAdornment position="end">
              {props.error ? (
                <ErrorRounded sx={{ color: "error.main" }} />
              ) : (
                <CheckCircle sx={{ color: "secondary.light" }} />
              )}
            </InputAdornment>
          ) : isPassword ? (
            <InputAdornment position="end" disabled={props.disabled}>
              <IconButton
                tabIndex={-1}
                disabled={props.disabled}
                onClick={() => setShow(!show)}
                onMouseDown={handleMouseDownPassword}
              >
                {show ? (
                  <Visibility
                    sx={
                      props.disabled
                        ? { color: "button.textDisabled" }
                        : { color: "primary.main" }
                    }
                    alt={`hide_${iscvvStyle ? `cvv` : `password`}`}
                  />
                ) : (
                  <VisibilityOff
                    sx={
                      props.disabled
                        ? { color: "button.textDisabled" }
                        : { color: "primary.main" }
                    }
                    alt={`show_${iscvvStyle ? `cvv` : `password`}`}
                  />
                )}
              </IconButton>
            </InputAdornment>
          ) : null,
          sx: {
            "&.MuiInputBase-root.MuiFilledInput-root": {
              backgroundColor: `#FFFFFF`,
              "&.Mui-disabled": {
                backgroundColor: `#EBEBEDCC`,
              },
            },
            border: `1px solid #B3B9C8`,
            borderRadius: `4px`,
            ".MuiInputBase-input": {
              color: color ? color : "black",
              paddingTop: label ? "16px" : iscvvStyle ? "14px" : "9px",
              paddingBottom: label ? "4px" : iscvvStyle ? "14px" : "11px",
            },
            "&.Mui-error": {
              border: `1px solid`,
              borderColor: "error.main",
            },
            ":before": {
              borderBottom: `none!important`,
            },
            ":after": {
              border: `none`,
            },
            ".Mui-focused": {
              backgroundColor: `#FFFFFF`,
            },
            ".Mui-disabled": {
              backgroundColor: `#EBEBEDCC`,
              WebkitTextFillColor: "#2D2D2F !important",
            },
            "input::-ms-reveal": {
              display: "none",
            },
            "input::-ms-clear": {
              display: "none",
            },
            ".MuiInputBase-root": {
              color: color ? color : "black",
              paddingTop: label ? "16px" : iscvvStyle ? "14px" : "9px",
              paddingBottom: label ? "4px" : iscvvStyle ? "14px" : "11px",
              backgroundColor: palette.text.alt,
              borderRadius: "4px",
              border: `1px solid ${palette.text.disabled}`,
              padding: "16px 32px 4px 12px !important",
              ".MuiInputBase-input": {
                padding: "0 !important",
              },
              "&:hover": {
                borderBottom: `1px solid ${palette.text.disabled}`,
              },
              "&::before": {
                borderBottom: 0,
              },
              "&:hover:before": {
                borderBottom: 0,
              },
              "&:after": {
                borderBottom: 0,
              },
              "&.Mui-error": {
                border: "1px solid",
                borderColor: "error.main",
              },
            },
            ".MuiInputBase-root:hover:not(.Mui-disbaled, .Mui-error):before": {
              borderBottom: "0",
            },
          },
        }}
        sx={{
          input: {
            fontStyle: `normal`,
            fontWeight: `400`,
            fontSize: `14px`,
            lineHeight: `24px`,
            color: `#2D2D2F`,
            "&::placeholder": {
              color: iscvvStyle && "#646464",
              opacity: iscvvStyle && 1,
            },
          },
        }}
        {...props}
      />
    </FormControl>
  );
}
